#ifndef __CString__
#define __CString__

//	===========================================================================

#ifdef WIN32
	#pragma warning (disable : 4522)
#endif

//	===========================================================================

#include "CCountedObject.hpp"
#include "CPascalString.hpp"

//	===========================================================================

using Exponent::Basics::CCountedObject;
using Exponent::Basics::CPascalString;

//	===========================================================================

namespace Exponent
{
	namespace Basics
	{
		/**
		 * @class CString CString.hpp
		 * @brief String handling class
		 *
		 * A string handling class. Takes care of most string handling duties,
		 * however, if you require path handling functions as well, please see
		 * @see CSystemString
		 *
		 * @date 21/08/2004
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 * @version 1.0.1 Fixed const correctness for operators
		 * @version 1.0.2 Added equalsIgnoringCase functions
		 * @version 1.0.3 Added replaceCharWithChar function
		 * @version 1.0.4 Added setStringWithFormat function
		 * @version 1.0.5 Added replaceCharWithString and replaceStringWithChar function
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 *
		 * $Id: CString.hpp,v 1.5 2007/02/08 21:06:44 paul Exp $
		 */
		class CString : public CCountedObject
		{
			/** @cond */
			EXPONENT_CLASS_DECLARATION;
			/** @endcond */

//	===========================================================================

		public:

//	===========================================================================

			const static long CSTRING_SHORT_TEXT_LENGTH  = 8;		/**< A Short string */
			const static long CSTRING_MIDDLE_TEXT_LENGTH = 256;		/**< A Medium string */
			const static long CSTRING_LONG_TEXT_LENGTH   = 1024;	/**< A Long string */
			const static char CSTRING_EMPTY_STRING[3];				/**< An empty string (set to " ") */
			const static char CSTRING_NULL_STRING[];				/**< A null string (set to "\0") */

//	===========================================================================

			/**
			 * Construction with a C-style string
			 * @param string The string you wish to store
			 */
			CString(const char *string = " ");

			/**
			 * Copy construction
			 * @param string The string to copy
			 */
			CString(const CString &string);

			/**
			 * Destruction
			 */
			virtual ~CString();

//	===========================================================================

			/**
			 * Assignment operator
			 * @param string The string to copy
			 * @retval CString& A reference to this
			 */
			virtual CString &operator = (const CString &string);

			/**
			 * Assignment operator
			 * @param string The string to copy
			 * @retval CString& A reference to this
			 */
			virtual CString &operator = (CString &string);

			/**
			 * Assignment operator
			 * @param string The string to copy
			 * @retval CString& A reference to this
			 */
			virtual CString &operator = (const char *string);

			/**
			 * Assignment operator
			 * @param value The value to copy
			 * @retval CString& A reference to this
			 */
			virtual CString &operator = (const float value);

			/**
			 * Assignment operator
			 * @param value The value to copy
			 * @retval CString& A reference to this
			 */
			virtual CString &operator = (const long value);

			/**
			 * Assignment operator
			 * @param value The value to copy
			 * @retval CString& A reference to this
			 */
			virtual CString &operator = (const int value);

			/**
			 * Assignment operator Sets the string to "True" or "False"
			 * @param value The value to copy
			 * @retval CString& A reference to this
			 */
			virtual CString &operator = (const bool value);

			/**
			 * Concatenate two strings
			 * @param string The string to append to the end of this
			 * @retval CString& A reference to this
			 */
			virtual CString &operator += (const CString &string);

			/**
			 * Concatenate two strings
			 * @param string The string to append to the end of this
			 * @retval CString& A reference to this
			 */
			virtual CString &operator += (const char *string);

			/**
			 * Equality operator
			 * @param equals The string to compare against
			 * @retval bool True if strings are equal, false otherwise
			 */
			virtual bool operator == (const char *equals) const;

			/**
			 * Equality operator
			 * @param string The string to compare against
			 * @retval bool True if strings are equal, false otherwise
			 */
			virtual bool operator == (const CString &string) const;

			/**
			 * Inequality operator
			 * @param string The string to compare against
			 * @retval bool True if strings are not equal, false otherwise
			 */
			virtual bool operator != (const char *string) const;

			/**
			 * Inequality operator
			 * @param string The string to compare against
			 * @retval bool True if strings are not equal, false otherwise
			 */
			virtual bool operator != (const CString &string) const;

			/**
			 * Check if the string is empty
			 * @retval bool true if string is empty, false otherwise
			 */
			virtual bool operator ! () const;

			/**
			 * Subscript operator
			 * @param index The index of the character you would like
			 * @retval char The character at the index
			 */
			virtual char operator [] (const long index) const;

//	===========================================================================

			/**
			 * Convert the string to a pascal string
			 * @retval CPascalString* The String as a pascal string.\n
			 * You are responsible for deleting the returned string
			 */
			virtual CPascalString getStringAsPascalString();

//  ===========================================================================

			/**
			 * Convert the text string to a double
			 * @param text The string to be converted
			 * @retval double The string as a double, or 0 if failure
			 */
			static double toDouble(const char *text);

			/**
			 * Convert the text string to a float
			 * @param text The string to be converted
			 * @retval float The string as a float, or 0 if failure
			 */
			static float toFloat(const char *text);

			/**
			 * Convert the text string to a long
			 * @param text The string to be converted
			 * @retval long The string as a long, or 0 if failure
			 */
			static long toLong(const char *text);

			/**
			 * Convert the text string to a int
			 * @param text The string to be converted
			 * @retval int The string as an int, or 0 if failure
			 */
			static int toInt(const char *text);

			/**
			 * Convert a float to a string
			 * @param value The value to be converted
			 * @retval char* The coverted string. You are responsible for deleting memory on returned object
			 */
			static char *toString(const float value);

			/**
			 * Convert an int to a string
			 * @param value The value to be converted
			 * @retval char* The coverted string. You are responsible for deleting memory on returned object
			 */
			static char *toString(const int value);

			/**
			 * Convert a long to a string
			 * @param value The value to be converted
			 * @retval char* The coverted string. You are responsible for deleting memory on returned object
			 */
			static char *toString(const long value);

			/**
			 * Get a copy of the empty string
			 * @retval CString* The empty string. You are responsible for deleting memory on returned object
			 */
			static CString *newEmptyString() { return new CString(CString::CSTRING_NULL_STRING); }

			/**
			 * Get a new empty string
			 * @retval CString The empty string
			 */
			static CString emptyString() { return CString(CString::CSTRING_NULL_STRING); }

//	===========================================================================

			/**
			 * Set the string
			 * @param string The string to store
			 */
			virtual void setString(const char *string);

			/**
			 * Set the string
			 * @param string The string to store
			 */
			virtual void setString(const CString &string);

			/**
			 * Format a text string - Used in an sprintf style
			 * @param text The text
			 * @param ... The variables
			 */
			virtual void setStringWithFormat(const char *text, ...);

//  ===========================================================================

			/**
			 * Append a single character to the end of this
			 * @param character The character to append
			 */
			virtual void appendString(const char character);

			/**
			 * Append a string to the end of this
			 * @param string The stirng to append
			 */
			virtual void appendString(const char *string);

			/**
			 * Append a string to the end of this
			 * @param string The stirng to append
			 */
			virtual void appendString(const CString &string);

//  ===========================================================================

			/**
			 * Get a copy of the string
			 * @param buffer The buffer to fill
			 * @param size The size of the buffer
			 */
			virtual void getString(char *buffer, const long size) const;

			/**
			 * Get a copy of the string
			 * @param buffer The buffer to fill (filled to CSTRING_SHORT_TEXT_LENGTH)
			 */
			virtual void getString(char *buffer) const;

			/**
			 * Get a sub string from the main string
			 * @param start The starting index
			 * @param end The ending index (is included in sub string)
			 * @retval CString A new sub string
			 */
			virtual CString getSubString(const long start, const long end) const;

			/**
			 * Get a pointer to the string
			 * @retval const char* A pointer to the string
			 */
			virtual const char *getString() const;

//  ===========================================================================

			/**
			 * Get the number of characters in the string
			 * @retval long The number of characters in the string
			 */
			virtual long getNumberOfCharacters() const;

			/**
			 * Get the character at a specific index
			 * @param index The index of the character you want
			 * @retval char The character at the index you specified
			 */
			virtual char characterAt(const int index) const;

//  ===========================================================================

			/**
			 * Converts the string to all uppercase
			 */
			virtual void toUpperCase();

			/**
			 * Converts the string to all lowercase
			 */
			virtual void toLowerCase();

//  ===========================================================================

			/**
			 * Remove trailing and leading white space
			 */
			virtual void removeTrailingAndLeadingWhiteSpace();

			/**
			 * Remove leading white space
			 */
			virtual void removeLeadingWhiteSpace();

			/**
			 * Remove trailing white space
			 */
			virtual void removeTrailingWhiteSpace();

			/**
			 * Delete the last character
			 */
			virtual void removeLastCharacter();

			/**
			 * Replace a character with another character
			 * @param characterToReplace The character that will be replaced
			 * @param characterToReplaceWith The character that will be used in the replacement
			 */
			virtual void replaceCharWithChar(const char characterToReplace, const char characterToReplaceWith);

			/**
			 * Replace a character with a string
			 * @param characterToReplace The character that will be replaced
			 * @param stringToReplaceWith The string to replace with
			 */
			virtual void replaceCharWithString(const char characterToReplace, const CString &stringToReplaceWith);

			/**
			 * Replace a given sub string with a character
			 * @param stringToReplace The string that will be replaced
			 * @param characterToReplaceWith The character to replace with
			 */
			virtual void replaceStringWithChar(const char characterToReplaceWith, const CString &stringToReplace);

//  ===========================================================================

			/**
			 * Change the case of a single character
			 * @param letter The letter to change the case of
			 * @retval char The character in upper case
			 */
			static char toUpperCase(const char letter);

			/**
			 * Change the case of a single character
			 * @param letter The letter to change the case of
			 * @retval char The character in lower case
			 */
			static char toLowerCase(const char letter);

			/**
			 * Check if the character is a white space
			 * @param letter The character to check
			 * @retval bool True if whitespace, false otherwise
			 */
			static bool isWhiteSpace(const char letter);

//  ===========================================================================

			/**
			 * Do the strings match case important
			 * @param other The string to compare against
			 * @retval bool True if strings are equal, false otherwise
			 */
			virtual bool equals(const char *other) const;

			/**
			 * Do the strings match case important
			 * @param string The string to compare against
			 * @retval bool True if strings are equal, false otherwise
			 */
			virtual bool equals(const CString &string) const;

			/**
			 * Do the strings match ignoring case
			 * @param other The string to compare against
			 * @retval bool True if strings are equal, false otherwise
			 */
			virtual bool equalsIgnoringCase(const char *other) const;

			/**
			 * Do the strings match ignoring case
			 * @param string The string to compare against
			 * @retval bool True if strings are equal, false otherwise
			 */
			virtual bool equalsIgnoringCase(const CString &string) const;

//  ===========================================================================

			/**
			 * Find the first occurance of a character
			 * @param character The character to search for
			 * @retval long -1 on failure, otherwise the index of first found
			 */
			virtual long findForward(const char character);

			/**
			 * Find the last occurance of a character
			 * @param character The character to search for
			 * @retval long -1 on failure, otherwise the index of last found
			 */
			virtual long findBackwards(const char character);

			/**
			 * Find a sub string
			 * @param string The string to find
			 * @retval long starting index of the sub string or -1 on error
			 */
			long findSubString(const CString &string);

//  ===========================================================================

#ifndef WIN32
			/**
			 * Get the string as a CFStringRef - MacOsX only
			 * @retval CFStringRef the CString as a CFStringRef.\n
			 * It is up to you to delete the memory used by this CFStringRef
			 */
			CFStringRef getStringAsCFStringRef() const;

			/**
			 * Set the string from a CFStringRef
			 * @param string The CFStringRef to copy
			 */
			virtual void setStringFromCFStringRef(const CFStringRef &string);
#endif

//	===========================================================================

			/**
			 * Get a description of the object
			 * @param string On return is filled with the description
			 * @param size The size of the stirng
			 */
			virtual void getObjectDescription(char *string, const long size) const;

//	===========================================================================

		protected:

//  ===========================================================================

			char *m_string;					/**< The string */
			long m_numberOfCharacters;		/**< The number of characters, not including terminator */
		};
	}
}
#endif	// End of CString.hpp